/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Deque;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.metrics.MeasurableStat;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.Avg;
import org.apache.kafka.common.metrics.stats.Rate;
import org.apache.kafka.common.metrics.stats.SampledStat;
import org.apache.kafka.common.metrics.stats.WindowedCount;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.streams.StreamsConfig;
import org.apache.kafka.streams.errors.StreamsException;
import org.apache.kafka.streams.errors.TaskCorruptedException;
import org.apache.kafka.streams.processor.TaskId;
import org.apache.kafka.streams.processor.internals.ChangelogReader;
import org.apache.kafka.streams.processor.internals.ReadOnlyTask;
import org.apache.kafka.streams.processor.internals.StandbyTask;
import org.apache.kafka.streams.processor.internals.StateUpdater;
import org.apache.kafka.streams.processor.internals.StreamTask;
import org.apache.kafka.streams.processor.internals.Task;
import org.apache.kafka.streams.processor.internals.TaskAndAction;
import org.apache.kafka.streams.processor.internals.TopologyMetadata;
import org.slf4j.Logger;

public class DefaultStateUpdater
implements StateUpdater {
    private static final String BUG_ERROR_MESSAGE = "This indicates a bug. Please report at https://issues.apache.org/jira/projects/KAFKA/issues or to the dev-mailing list (https://kafka.apache.org/contact).";
    private final Time time;
    private final Logger log;
    private final String name;
    private final Metrics metrics;
    private final ChangelogReader changelogReader;
    private final TopologyMetadata topologyMetadata;
    private final Queue<TaskAndAction> tasksAndActions = new LinkedList<TaskAndAction>();
    private final Lock tasksAndActionsLock = new ReentrantLock();
    private final Condition tasksAndActionsCondition = this.tasksAndActionsLock.newCondition();
    private final Queue<StreamTask> restoredActiveTasks = new LinkedList<StreamTask>();
    private final Lock restoredActiveTasksLock = new ReentrantLock();
    private final Condition restoredActiveTasksCondition = this.restoredActiveTasksLock.newCondition();
    private final BlockingQueue<StateUpdater.ExceptionAndTasks> exceptionsAndFailedTasks = new LinkedBlockingQueue<StateUpdater.ExceptionAndTasks>();
    private final BlockingQueue<Task> removedTasks = new LinkedBlockingQueue<Task>();
    private final AtomicBoolean isTopologyResumed = new AtomicBoolean(false);
    private final long commitIntervalMs;
    private long lastCommitMs;
    private StateUpdaterThread stateUpdaterThread = null;
    private CountDownLatch shutdownGate;

    public DefaultStateUpdater(String name, Metrics metrics, StreamsConfig config, ChangelogReader changelogReader, TopologyMetadata topologyMetadata, Time time) {
        this.time = time;
        this.name = name;
        this.metrics = metrics;
        this.changelogReader = changelogReader;
        this.topologyMetadata = topologyMetadata;
        this.commitIntervalMs = config.getLong("commit.interval.ms");
        String logPrefix = String.format("state-updater [%s] ", name);
        LogContext logContext = new LogContext(logPrefix);
        this.log = logContext.logger(DefaultStateUpdater.class);
    }

    @Override
    public void start() {
        if (this.stateUpdaterThread == null) {
            this.stateUpdaterThread = new StateUpdaterThread(this.name, this.metrics, this.changelogReader);
            this.stateUpdaterThread.start();
            this.shutdownGate = new CountDownLatch(1);
            this.lastCommitMs = this.time.milliseconds();
        }
    }

    @Override
    public void shutdown(Duration timeout) {
        if (this.stateUpdaterThread != null) {
            this.log.info("Shutting down state updater thread");
            this.stateUpdaterThread.isRunning.set(false);
            this.tasksAndActionsLock.lock();
            try {
                this.tasksAndActionsCondition.signalAll();
            }
            finally {
                this.tasksAndActionsLock.unlock();
            }
            try {
                if (!this.shutdownGate.await(timeout.toMillis(), TimeUnit.MILLISECONDS)) {
                    throw new StreamsException("State updater thread did not shutdown within the timeout");
                }
                this.stateUpdaterThread = null;
            }
            catch (InterruptedException interruptedException) {}
        } else {
            this.removeAddedTasksFromInputQueue();
        }
    }

    private void removeAddedTasksFromInputQueue() {
        this.tasksAndActionsLock.lock();
        try {
            TaskAndAction taskAndAction;
            while ((taskAndAction = this.tasksAndActions.peek()) != null) {
                if (taskAndAction.getAction() == TaskAndAction.Action.ADD) {
                    this.removedTasks.add(taskAndAction.getTask());
                }
                this.tasksAndActions.poll();
            }
        }
        finally {
            this.tasksAndActionsLock.unlock();
        }
    }

    @Override
    public void add(Task task) {
        this.verifyStateFor(task);
        this.tasksAndActionsLock.lock();
        try {
            this.tasksAndActions.add(TaskAndAction.createAddTask(task));
            this.tasksAndActionsCondition.signalAll();
        }
        finally {
            this.tasksAndActionsLock.unlock();
        }
    }

    private void verifyStateFor(Task task) {
        if (task.isActive() && task.state() != Task.State.RESTORING) {
            throw new IllegalStateException("Active task " + task.id() + " is not in state RESTORING. " + BUG_ERROR_MESSAGE);
        }
        if (!task.isActive() && task.state() != Task.State.RUNNING) {
            throw new IllegalStateException("Standby task " + task.id() + " is not in state RUNNING. " + BUG_ERROR_MESSAGE);
        }
    }

    @Override
    public void remove(TaskId taskId) {
        this.tasksAndActionsLock.lock();
        try {
            this.tasksAndActions.add(TaskAndAction.createRemoveTask(taskId));
            this.tasksAndActionsCondition.signalAll();
        }
        finally {
            this.tasksAndActionsLock.unlock();
        }
    }

    @Override
    public void signalResume() {
        this.tasksAndActionsLock.lock();
        try {
            this.isTopologyResumed.set(true);
            this.tasksAndActionsCondition.signalAll();
        }
        finally {
            this.tasksAndActionsLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Set<StreamTask> drainRestoredActiveTasks(Duration timeout) {
        long timeoutMs = timeout.toMillis();
        long startTime = this.time.milliseconds();
        long deadline = startTime + timeoutMs;
        long now = startTime;
        HashSet<StreamTask> result = new HashSet<StreamTask>();
        try {
            while (now <= deadline && result.isEmpty()) {
                this.restoredActiveTasksLock.lock();
                try {
                    while (this.restoredActiveTasks.isEmpty() && now <= deadline) {
                        boolean elapsed = this.restoredActiveTasksCondition.await(deadline - now, TimeUnit.MILLISECONDS);
                        now = this.time.milliseconds();
                    }
                    result.addAll(this.restoredActiveTasks);
                    this.restoredActiveTasks.clear();
                }
                finally {
                    this.restoredActiveTasksLock.unlock();
                }
                now = this.time.milliseconds();
            }
            return result;
        }
        catch (InterruptedException interruptedException) {
            return result;
        }
    }

    @Override
    public Set<Task> drainRemovedTasks() {
        ArrayList result = new ArrayList();
        this.removedTasks.drainTo(result);
        return new HashSet<Task>(result);
    }

    @Override
    public boolean hasRemovedTasks() {
        return !this.removedTasks.isEmpty();
    }

    @Override
    public List<StateUpdater.ExceptionAndTasks> drainExceptionsAndFailedTasks() {
        ArrayList<StateUpdater.ExceptionAndTasks> result = new ArrayList<StateUpdater.ExceptionAndTasks>();
        this.exceptionsAndFailedTasks.drainTo(result);
        return result;
    }

    @Override
    public boolean hasExceptionsAndFailedTasks() {
        return !this.exceptionsAndFailedTasks.isEmpty();
    }

    public Set<StandbyTask> getUpdatingStandbyTasks() {
        return this.stateUpdaterThread != null ? Collections.unmodifiableSet(new HashSet<StandbyTask>(this.stateUpdaterThread.getUpdatingStandbyTasks())) : Collections.emptySet();
    }

    @Override
    public Set<Task> getUpdatingTasks() {
        return this.stateUpdaterThread != null ? Collections.unmodifiableSet(new HashSet<Task>(this.stateUpdaterThread.getUpdatingTasks())) : Collections.emptySet();
    }

    public Set<StreamTask> getRestoredActiveTasks() {
        this.restoredActiveTasksLock.lock();
        try {
            Set<StreamTask> set = Collections.unmodifiableSet(new HashSet<StreamTask>(this.restoredActiveTasks));
            return set;
        }
        finally {
            this.restoredActiveTasksLock.unlock();
        }
    }

    public List<StateUpdater.ExceptionAndTasks> getExceptionsAndFailedTasks() {
        return Collections.unmodifiableList(new ArrayList<StateUpdater.ExceptionAndTasks>(this.exceptionsAndFailedTasks));
    }

    public Set<Task> getRemovedTasks() {
        return Collections.unmodifiableSet(new HashSet<Task>(this.removedTasks));
    }

    public Set<Task> getPausedTasks() {
        return this.stateUpdaterThread != null ? Collections.unmodifiableSet(new HashSet<Task>(this.stateUpdaterThread.getPausedTasks())) : Collections.emptySet();
    }

    @Override
    public Set<Task> getTasks() {
        return this.executeWithQueuesLocked(() -> this.getStreamOfTasks().map(ReadOnlyTask::new).collect(Collectors.toSet()));
    }

    @Override
    public boolean restoresActiveTasks() {
        return !this.executeWithQueuesLocked(() -> this.getStreamOfTasks().filter(Task::isActive).collect(Collectors.toSet())).isEmpty();
    }

    public Set<StreamTask> getActiveTasks() {
        return this.executeWithQueuesLocked(() -> this.getStreamOfTasks().filter(Task::isActive).map(t -> (StreamTask)t).collect(Collectors.toSet()));
    }

    @Override
    public Set<StandbyTask> getStandbyTasks() {
        return this.executeWithQueuesLocked(() -> this.getStreamOfTasks().filter(t -> !t.isActive()).map(t -> (StandbyTask)t).collect(Collectors.toSet()));
    }

    private <T> Set<T> executeWithQueuesLocked(Supplier<Set<T>> action) {
        this.tasksAndActionsLock.lock();
        this.restoredActiveTasksLock.lock();
        try {
            Set<T> set = action.get();
            return set;
        }
        finally {
            this.restoredActiveTasksLock.unlock();
            this.tasksAndActionsLock.unlock();
        }
    }

    private Stream<Task> getStreamOfTasks() {
        return Stream.concat(this.getStreamOfNonPausedTasks(), this.getPausedTasks().stream());
    }

    private Stream<Task> getStreamOfNonPausedTasks() {
        return Stream.concat(this.tasksAndActions.stream().filter(taskAndAction -> taskAndAction.getAction() == TaskAndAction.Action.ADD).map(TaskAndAction::getTask), Stream.concat(this.getUpdatingTasks().stream(), Stream.concat(this.restoredActiveTasks.stream(), Stream.concat(this.exceptionsAndFailedTasks.stream().flatMap(exceptionAndTasks -> exceptionAndTasks.getTasks().stream()), this.removedTasks.stream()))));
    }

    private class StateUpdaterThread
    extends Thread {
        private final ChangelogReader changelogReader;
        private final StateUpdaterMetrics updaterMetrics;
        private final AtomicBoolean isRunning;
        private final Map<TaskId, Task> updatingTasks;
        private final Map<TaskId, Task> pausedTasks;
        private long totalCheckpointLatency;

        public StateUpdaterThread(String name, Metrics metrics, ChangelogReader changelogReader) {
            super(name);
            this.isRunning = new AtomicBoolean(true);
            this.updatingTasks = new ConcurrentHashMap<TaskId, Task>();
            this.pausedTasks = new ConcurrentHashMap<TaskId, Task>();
            this.totalCheckpointLatency = 0L;
            this.changelogReader = changelogReader;
            this.updaterMetrics = new StateUpdaterMetrics(metrics, name);
        }

        public Collection<Task> getUpdatingTasks() {
            return this.updatingTasks.values();
        }

        public Collection<StandbyTask> getUpdatingStandbyTasks() {
            return this.updatingTasks.values().stream().filter(t -> !t.isActive()).map(t -> (StandbyTask)t).collect(Collectors.toList());
        }

        private boolean onlyStandbyTasksUpdating() {
            return !this.updatingTasks.isEmpty() && this.updatingTasks.values().stream().noneMatch(Task::isActive);
        }

        public Collection<Task> getPausedTasks() {
            return this.pausedTasks.values();
        }

        public long getNumUpdatingStandbyTasks() {
            return this.updatingTasks.values().stream().filter(t -> !t.isActive()).count();
        }

        public long getNumRestoringActiveTasks() {
            return this.updatingTasks.values().stream().filter(Task::isActive).count();
        }

        public long getNumPausedStandbyTasks() {
            return this.pausedTasks.values().stream().filter(t -> !t.isActive()).count();
        }

        public long getNumPausedActiveTasks() {
            return this.pausedTasks.values().stream().filter(Task::isActive).count();
        }

        @Override
        public void run() {
            DefaultStateUpdater.this.log.info("State updater thread started");
            try {
                while (this.isRunning.get()) {
                    this.runOnce();
                }
            }
            catch (RuntimeException anyOtherException) {
                this.handleRuntimeException(anyOtherException);
            }
            finally {
                DefaultStateUpdater.this.removeAddedTasksFromInputQueue();
                this.removeUpdatingAndPausedTasks();
                this.updaterMetrics.clear();
                DefaultStateUpdater.this.shutdownGate.countDown();
                DefaultStateUpdater.this.log.info("State updater thread stopped");
            }
        }

        private void runOnce() {
            long totalStartTimeMs = DefaultStateUpdater.this.time.milliseconds();
            this.performActionsOnTasks();
            this.resumeTasks();
            this.pauseTasks();
            this.restoreTasks(totalStartTimeMs);
            long checkpointStartTimeMs = DefaultStateUpdater.this.time.milliseconds();
            this.maybeCheckpointTasks(checkpointStartTimeMs);
            long waitStartTimeMs = DefaultStateUpdater.this.time.milliseconds();
            this.waitIfAllChangelogsCompletelyRead();
            long endTimeMs = DefaultStateUpdater.this.time.milliseconds();
            long totalWaitTime = Math.max(0L, endTimeMs - waitStartTimeMs);
            long totalTime = Math.max(0L, endTimeMs - totalStartTimeMs);
            this.recordMetrics(endTimeMs, totalTime, totalWaitTime);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void performActionsOnTasks() {
            DefaultStateUpdater.this.tasksAndActionsLock.lock();
            try {
                for (TaskAndAction taskAndAction : this.getTasksAndActions()) {
                    TaskAndAction.Action action = taskAndAction.getAction();
                    switch (action) {
                        case ADD: {
                            this.addTask(taskAndAction.getTask());
                            break;
                        }
                        case REMOVE: {
                            this.removeTask(taskAndAction.getTaskId());
                        }
                    }
                }
            }
            finally {
                DefaultStateUpdater.this.tasksAndActionsLock.unlock();
            }
        }

        private void resumeTasks() {
            if (DefaultStateUpdater.this.isTopologyResumed.compareAndSet(true, false)) {
                for (Task task : this.pausedTasks.values()) {
                    if (DefaultStateUpdater.this.topologyMetadata.isPaused(task.id().topologyName())) continue;
                    this.resumeTask(task);
                }
            }
        }

        private void pauseTasks() {
            for (Task task : this.updatingTasks.values()) {
                if (!DefaultStateUpdater.this.topologyMetadata.isPaused(task.id().topologyName())) continue;
                this.pauseTask(task);
            }
        }

        private void restoreTasks(long now) {
            try {
                long restored = this.changelogReader.restore(this.updatingTasks);
                this.updaterMetrics.restoreSensor.record((double)restored, now);
            }
            catch (TaskCorruptedException taskCorruptedException) {
                this.handleTaskCorruptedException(taskCorruptedException);
            }
            catch (StreamsException streamsException) {
                this.handleStreamsException(streamsException);
            }
            Set<TopicPartition> completedChangelogs = this.changelogReader.completedChangelogs();
            List activeTasks = this.updatingTasks.values().stream().filter(Task::isActive).collect(Collectors.toList());
            for (Task task : activeTasks) {
                this.maybeCompleteRestoration((StreamTask)task, completedChangelogs);
            }
        }

        private void handleRuntimeException(RuntimeException runtimeException) {
            DefaultStateUpdater.this.log.error("An unexpected error occurred within the state updater thread: " + runtimeException);
            this.addToExceptionsAndFailedTasksThenClearUpdatingTasks(new StateUpdater.ExceptionAndTasks(new HashSet<Task>(this.updatingTasks.values()), runtimeException));
            this.isRunning.set(false);
        }

        private void handleTaskCorruptedException(TaskCorruptedException taskCorruptedException) {
            DefaultStateUpdater.this.log.info("Encountered task corrupted exception: ", (Throwable)((Object)taskCorruptedException));
            Set<TaskId> corruptedTaskIds = taskCorruptedException.corruptedTasks();
            HashSet<Task> corruptedTasks = new HashSet<Task>();
            HashSet<TopicPartition> changelogsOfCorruptedTasks = new HashSet<TopicPartition>();
            for (TaskId taskId : corruptedTaskIds) {
                Task corruptedTask = this.updatingTasks.get(taskId);
                if (corruptedTask == null) {
                    throw new IllegalStateException("Task " + taskId + " is corrupted but is not updating. " + DefaultStateUpdater.BUG_ERROR_MESSAGE);
                }
                corruptedTasks.add(corruptedTask);
                this.removeCheckpointForCorruptedTask(corruptedTask);
                changelogsOfCorruptedTasks.addAll(corruptedTask.changelogPartitions());
            }
            this.changelogReader.unregister(changelogsOfCorruptedTasks);
            this.addToExceptionsAndFailedTasksThenRemoveFromUpdatingTasks(new StateUpdater.ExceptionAndTasks(corruptedTasks, (RuntimeException)((Object)taskCorruptedException)));
        }

        private void removeCheckpointForCorruptedTask(Task task) {
            task.markChangelogAsCorrupted(task.changelogPartitions());
            this.measureCheckpointLatency(() -> task.maybeCheckpoint(true));
        }

        private void handleStreamsException(StreamsException streamsException) {
            DefaultStateUpdater.this.log.info("Encountered streams exception: ", (Throwable)((Object)streamsException));
            if (streamsException.taskId().isPresent()) {
                this.handleStreamsExceptionWithTask(streamsException);
            } else {
                this.handleStreamsExceptionWithoutTask(streamsException);
            }
        }

        private void handleStreamsExceptionWithTask(StreamsException streamsException) {
            TaskId failedTaskId = streamsException.taskId().get();
            if (!this.updatingTasks.containsKey(failedTaskId)) {
                throw new IllegalStateException("Task " + failedTaskId + " failed but is not updating. " + DefaultStateUpdater.BUG_ERROR_MESSAGE);
            }
            HashSet<Task> failedTask = new HashSet<Task>();
            failedTask.add(this.updatingTasks.get(failedTaskId));
            this.addToExceptionsAndFailedTasksThenRemoveFromUpdatingTasks(new StateUpdater.ExceptionAndTasks(failedTask, (RuntimeException)((Object)streamsException)));
        }

        private void handleStreamsExceptionWithoutTask(StreamsException streamsException) {
            this.addToExceptionsAndFailedTasksThenClearUpdatingTasks(new StateUpdater.ExceptionAndTasks(new HashSet<Task>(this.updatingTasks.values()), (RuntimeException)((Object)streamsException)));
        }

        private void addToExceptionsAndFailedTasksThenRemoveFromUpdatingTasks(StateUpdater.ExceptionAndTasks exceptionAndTasks) {
            DefaultStateUpdater.this.exceptionsAndFailedTasks.add(exceptionAndTasks);
            exceptionAndTasks.getTasks().stream().map(Task::id).forEach(this.updatingTasks::remove);
            if (exceptionAndTasks.getTasks().stream().anyMatch(Task::isActive)) {
                this.transitToUpdateStandbysIfOnlyStandbysLeft();
            }
        }

        private void addToExceptionsAndFailedTasksThenClearUpdatingTasks(StateUpdater.ExceptionAndTasks exceptionAndTasks) {
            DefaultStateUpdater.this.exceptionsAndFailedTasks.add(exceptionAndTasks);
            this.updatingTasks.clear();
        }

        private void waitIfAllChangelogsCompletelyRead() {
            DefaultStateUpdater.this.tasksAndActionsLock.lock();
            try {
                while (this.isRunning.get() && this.changelogReader.allChangelogsCompleted() && DefaultStateUpdater.this.tasksAndActions.isEmpty() && !DefaultStateUpdater.this.isTopologyResumed.get()) {
                    DefaultStateUpdater.this.tasksAndActionsCondition.await();
                }
            }
            catch (InterruptedException interruptedException) {
            }
            finally {
                DefaultStateUpdater.this.tasksAndActionsLock.unlock();
            }
        }

        private void removeUpdatingAndPausedTasks() {
            this.changelogReader.clear();
            this.measureCheckpointLatency(() -> this.updatingTasks.forEach((id, task) -> {
                task.maybeCheckpoint(true);
                DefaultStateUpdater.this.removedTasks.add(task);
            }));
            this.updatingTasks.clear();
            this.pausedTasks.forEach((id, task) -> DefaultStateUpdater.this.removedTasks.add(task));
            this.pausedTasks.clear();
        }

        private List<TaskAndAction> getTasksAndActions() {
            ArrayList<TaskAndAction> tasksAndActionsToProcess = new ArrayList<TaskAndAction>(DefaultStateUpdater.this.tasksAndActions);
            DefaultStateUpdater.this.tasksAndActions.clear();
            return tasksAndActionsToProcess;
        }

        private void addTask(Task task) {
            TaskId taskId = task.id();
            Task existingTask = this.pausedTasks.get(taskId);
            if (existingTask != null) {
                throw new IllegalStateException((existingTask.isActive() ? "Active" : "Standby") + " task " + taskId + " already exist in paused tasks, should not try to add another " + (task.isActive() ? "active" : "standby") + " task with the same id. " + DefaultStateUpdater.BUG_ERROR_MESSAGE);
            }
            existingTask = this.updatingTasks.get(taskId);
            if (existingTask != null) {
                throw new IllegalStateException((existingTask.isActive() ? "Active" : "Standby") + " task " + taskId + " already exist in updating tasks, should not try to add another " + (task.isActive() ? "active" : "standby") + " task with the same id. " + DefaultStateUpdater.BUG_ERROR_MESSAGE);
            }
            if (this.isStateless(task)) {
                this.addToRestoredTasks((StreamTask)task);
                DefaultStateUpdater.this.log.info("Stateless active task " + taskId + " was added to the restored tasks of the state updater");
            } else if (DefaultStateUpdater.this.topologyMetadata.isPaused(taskId.topologyName())) {
                this.pausedTasks.put(taskId, task);
                this.changelogReader.register(task.changelogPartitions(), task.stateManager());
                DefaultStateUpdater.this.log.debug((task.isActive() ? "Active" : "Standby") + " task " + taskId + " was directly added to the paused tasks.");
            } else {
                this.updatingTasks.put(taskId, task);
                this.changelogReader.register(task.changelogPartitions(), task.stateManager());
                if (task.isActive()) {
                    DefaultStateUpdater.this.log.info("Stateful active task " + taskId + " was added to the state updater");
                    this.changelogReader.enforceRestoreActive();
                } else {
                    DefaultStateUpdater.this.log.info("Standby task " + taskId + " was added to the state updater");
                    if (this.updatingTasks.size() == 1) {
                        this.changelogReader.transitToUpdateStandby();
                    }
                }
            }
        }

        private void removeTask(TaskId taskId) {
            if (this.updatingTasks.containsKey(taskId)) {
                Task task = this.updatingTasks.get(taskId);
                this.measureCheckpointLatency(() -> task.maybeCheckpoint(true));
                Set<TopicPartition> changelogPartitions = task.changelogPartitions();
                this.changelogReader.unregister(changelogPartitions);
                DefaultStateUpdater.this.removedTasks.add(task);
                this.updatingTasks.remove(taskId);
                if (task.isActive()) {
                    this.transitToUpdateStandbysIfOnlyStandbysLeft();
                }
                DefaultStateUpdater.this.log.info((task.isActive() ? "Active" : "Standby") + " task " + task.id() + " was removed from the updating tasks and added to the removed tasks.");
            } else if (this.pausedTasks.containsKey(taskId)) {
                Task task = this.pausedTasks.get(taskId);
                Set<TopicPartition> changelogPartitions = task.changelogPartitions();
                this.changelogReader.unregister(changelogPartitions);
                DefaultStateUpdater.this.removedTasks.add(task);
                this.pausedTasks.remove(taskId);
                DefaultStateUpdater.this.log.info((task.isActive() ? "Active" : "Standby") + " task " + task.id() + " was removed from the paused tasks and added to the removed tasks.");
            } else {
                DefaultStateUpdater.this.log.info("Task " + taskId + " was not removed since it is not updating or paused.");
            }
        }

        private void pauseTask(Task task) {
            TaskId taskId = task.id();
            this.measureCheckpointLatency(() -> task.maybeCheckpoint(true));
            this.pausedTasks.put(taskId, task);
            this.updatingTasks.remove(taskId);
            if (task.isActive()) {
                this.transitToUpdateStandbysIfOnlyStandbysLeft();
            }
            DefaultStateUpdater.this.log.info((task.isActive() ? "Active" : "Standby") + " task " + task.id() + " was paused from the updating tasks and added to the paused tasks.");
        }

        private void resumeTask(Task task) {
            TaskId taskId = task.id();
            this.updatingTasks.put(taskId, task);
            this.pausedTasks.remove(taskId);
            if (task.isActive()) {
                DefaultStateUpdater.this.log.info("Stateful active task " + task.id() + " was resumed to the updating tasks of the state updater");
                this.changelogReader.enforceRestoreActive();
            } else {
                DefaultStateUpdater.this.log.info("Standby task " + task.id() + " was resumed to the updating tasks of the state updater");
                if (this.updatingTasks.size() == 1) {
                    this.changelogReader.transitToUpdateStandby();
                }
            }
        }

        private boolean isStateless(Task task) {
            return task.changelogPartitions().isEmpty() && task.isActive();
        }

        private void maybeCompleteRestoration(StreamTask task, Set<TopicPartition> restoredChangelogs) {
            Set<TopicPartition> changelogPartitions = task.changelogPartitions();
            if (restoredChangelogs.containsAll(changelogPartitions)) {
                this.measureCheckpointLatency(() -> task.maybeCheckpoint(true));
                this.changelogReader.unregister(changelogPartitions);
                this.addToRestoredTasks(task);
                this.updatingTasks.remove(task.id());
                DefaultStateUpdater.this.log.info("Stateful active task " + task.id() + " completed restoration");
                this.transitToUpdateStandbysIfOnlyStandbysLeft();
            }
        }

        private void transitToUpdateStandbysIfOnlyStandbysLeft() {
            if (this.onlyStandbyTasksUpdating()) {
                this.changelogReader.transitToUpdateStandby();
            }
        }

        private void addToRestoredTasks(StreamTask task) {
            DefaultStateUpdater.this.restoredActiveTasksLock.lock();
            try {
                DefaultStateUpdater.this.restoredActiveTasks.add(task);
                DefaultStateUpdater.this.log.debug("Active task " + task.id() + " was added to the restored tasks");
                DefaultStateUpdater.this.restoredActiveTasksCondition.signalAll();
            }
            finally {
                DefaultStateUpdater.this.restoredActiveTasksLock.unlock();
            }
        }

        private void maybeCheckpointTasks(long now) {
            long elapsedMsSinceLastCommit = now - DefaultStateUpdater.this.lastCommitMs;
            if (elapsedMsSinceLastCommit > DefaultStateUpdater.this.commitIntervalMs) {
                if (DefaultStateUpdater.this.log.isDebugEnabled()) {
                    DefaultStateUpdater.this.log.debug("Checkpointing state of all restoring tasks since {}ms has elapsed (commit interval is {}ms)", (Object)elapsedMsSinceLastCommit, (Object)DefaultStateUpdater.this.commitIntervalMs);
                }
                this.measureCheckpointLatency(() -> {
                    for (Task task : this.updatingTasks.values()) {
                        task.maybeCheckpoint(false);
                    }
                });
                DefaultStateUpdater.this.lastCommitMs = now;
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void measureCheckpointLatency(Runnable actionToMeasure) {
            long startMs = DefaultStateUpdater.this.time.milliseconds();
            try {
                actionToMeasure.run();
            }
            finally {
                this.totalCheckpointLatency += Math.max(0L, DefaultStateUpdater.this.time.milliseconds() - startMs);
            }
        }

        private void recordMetrics(long now, long totalLatency, long totalWaitLatency) {
            long totalRestoreLatency = Math.max(0L, totalLatency - totalWaitLatency - this.totalCheckpointLatency);
            this.updaterMetrics.idleRatioSensor.record((double)totalWaitLatency / (double)totalLatency, now);
            this.updaterMetrics.checkpointRatioSensor.record((double)this.totalCheckpointLatency / (double)totalLatency, now);
            if (this.changelogReader.isRestoringActive()) {
                this.updaterMetrics.activeRestoreRatioSensor.record((double)totalRestoreLatency / (double)totalLatency, now);
                this.updaterMetrics.standbyRestoreRatioSensor.record(0.0, now);
            } else {
                this.updaterMetrics.standbyRestoreRatioSensor.record((double)totalRestoreLatency / (double)totalLatency, now);
                this.updaterMetrics.activeRestoreRatioSensor.record(0.0, now);
            }
            this.totalCheckpointLatency = 0L;
        }
    }

    private class StateUpdaterMetrics {
        private static final String STATE_LEVEL_GROUP = "stream-state-updater-metrics";
        private static final String IDLE_RATIO_DESCRIPTION = "The fraction of time the thread spent on being idle";
        private static final String RESTORE_RATIO_DESCRIPTION = "The fraction of time the thread spent on restoring active tasks";
        private static final String UPDATE_RATIO_DESCRIPTION = "The fraction of time the thread spent on updating standby tasks";
        private static final String CHECKPOINT_RATIO_DESCRIPTION = "The fraction of time the thread spent on checkpointing tasks restored progress";
        private static final String RESTORE_RECORDS_RATE_DESCRIPTION = "The average per-second number of records restored";
        private static final String RESTORE_RATE_DESCRIPTION = "The average per-second number of restore calls triggered";
        private final Sensor restoreSensor;
        private final Sensor idleRatioSensor;
        private final Sensor activeRestoreRatioSensor;
        private final Sensor standbyRestoreRatioSensor;
        private final Sensor checkpointRatioSensor;
        private final Deque<String> allSensorNames = new LinkedList<String>();
        private final Deque<MetricName> allMetricNames = new LinkedList<MetricName>();

        private StateUpdaterMetrics(Metrics metrics, String threadId) {
            LinkedHashMap<String, String> threadLevelTags = new LinkedHashMap<String, String>();
            threadLevelTags.put("thread-id", threadId);
            MetricName metricName = metrics.metricName("active-restoring-tasks", STATE_LEVEL_GROUP, "The number of active tasks currently undergoing restoration", threadLevelTags);
            metrics.addMetric(metricName, (config, now) -> DefaultStateUpdater.this.stateUpdaterThread != null ? (double)DefaultStateUpdater.this.stateUpdaterThread.getNumRestoringActiveTasks() : 0.0);
            this.allMetricNames.push(metricName);
            metricName = metrics.metricName("standby-updating-tasks", STATE_LEVEL_GROUP, "The number of standby tasks currently undergoing state update", threadLevelTags);
            metrics.addMetric(metricName, (config, now) -> DefaultStateUpdater.this.stateUpdaterThread != null ? (double)DefaultStateUpdater.this.stateUpdaterThread.getNumUpdatingStandbyTasks() : 0.0);
            this.allMetricNames.push(metricName);
            metricName = metrics.metricName("active-paused-tasks", STATE_LEVEL_GROUP, "The number of active tasks paused restoring", threadLevelTags);
            metrics.addMetric(metricName, (config, now) -> DefaultStateUpdater.this.stateUpdaterThread != null ? (double)DefaultStateUpdater.this.stateUpdaterThread.getNumPausedActiveTasks() : 0.0);
            this.allMetricNames.push(metricName);
            metricName = metrics.metricName("standby-paused-tasks", STATE_LEVEL_GROUP, "The number of standby tasks paused state update", threadLevelTags);
            metrics.addMetric(metricName, (config, now) -> DefaultStateUpdater.this.stateUpdaterThread != null ? (double)DefaultStateUpdater.this.stateUpdaterThread.getNumPausedStandbyTasks() : 0.0);
            this.allMetricNames.push(metricName);
            this.idleRatioSensor = metrics.sensor("idle-ratio", Sensor.RecordingLevel.INFO);
            this.idleRatioSensor.add(new MetricName("idle-ratio", STATE_LEVEL_GROUP, IDLE_RATIO_DESCRIPTION, threadLevelTags), (MeasurableStat)new Avg());
            this.allSensorNames.add("idle-ratio");
            this.activeRestoreRatioSensor = metrics.sensor("active-restore-ratio", Sensor.RecordingLevel.INFO);
            this.activeRestoreRatioSensor.add(new MetricName("active-restore-ratio", STATE_LEVEL_GROUP, RESTORE_RATIO_DESCRIPTION, threadLevelTags), (MeasurableStat)new Avg());
            this.allSensorNames.add("active-restore-ratio");
            this.standbyRestoreRatioSensor = metrics.sensor("standby-update-ratio", Sensor.RecordingLevel.INFO);
            this.standbyRestoreRatioSensor.add(new MetricName("standby-update-ratio", STATE_LEVEL_GROUP, UPDATE_RATIO_DESCRIPTION, threadLevelTags), (MeasurableStat)new Avg());
            this.allSensorNames.add("standby-update-ratio");
            this.checkpointRatioSensor = metrics.sensor("checkpoint-ratio", Sensor.RecordingLevel.INFO);
            this.checkpointRatioSensor.add(new MetricName("checkpoint-ratio", STATE_LEVEL_GROUP, CHECKPOINT_RATIO_DESCRIPTION, threadLevelTags), (MeasurableStat)new Avg());
            this.allSensorNames.add("checkpoint-ratio");
            this.restoreSensor = metrics.sensor("restore-records", Sensor.RecordingLevel.INFO);
            this.restoreSensor.add(new MetricName("restore-records-rate", STATE_LEVEL_GROUP, RESTORE_RECORDS_RATE_DESCRIPTION, threadLevelTags), (MeasurableStat)new Rate());
            this.restoreSensor.add(new MetricName("restore-call-rate", STATE_LEVEL_GROUP, RESTORE_RATE_DESCRIPTION, threadLevelTags), (MeasurableStat)new Rate((SampledStat)new WindowedCount()));
            this.allSensorNames.add("restore-records");
        }

        void clear() {
            while (!this.allSensorNames.isEmpty()) {
                DefaultStateUpdater.this.metrics.removeSensor(this.allSensorNames.pop());
            }
            while (!this.allMetricNames.isEmpty()) {
                DefaultStateUpdater.this.metrics.removeMetric(this.allMetricNames.pop());
            }
        }
    }
}

